#include <stdio.h>
#include <mpi.h> 
#include <math.h>
/***********************************************************************
 * 
 * This file contains C code for modelling a simple predator-prey model.
 * The two animal populations modelled are rabbits and foxes that live on
 * some piece of land. The animal populations are represented by two two-
 * dimensional arrays of size NS_Size by WE_Size. Precisely, the number of
 * foxes that live in the i,j-th stretch of land (with 1<=i<=NS_Size and
 * 1<=j<=WE_Size) is stored as Fox[i][j]. The corresponding figure for
 * rabbits is stored in Rabbit[i][j]. Boundary conditions are enforced so
 * that the land is actually a torus and can be thought to extend
 * periodically beyond the given bounds NS_Size and WE_Size.  The
 * populations of each stretch is updated at each generation according to
 * some simple rules and the total populations are summed at regular
 * intervals of time.
 * 
 * Your first major task when parallelizing this code will be to define a
 * geometric data decomposition. Comments have been added to assist you.
 * They assume that the processes have been organised into a two
 * dimensional cartesian topology as shown in the diagram below. Arrows
 * point toward the direction of increasing indices.
 * 
 *
 *       +-----J----->       +-----Y----->
 *       |                   |              
 *       |                   |                  +--> East
 *       I  A[I][J]          X Proc[X][Y]       |        
 *       |                   |                  V 
 *       |                   |                South
 *       V                   V               
 *
 *
 * Some of the variables declared later are not needed in the sequential
 * code. They have been kept in case you find them useful when paralleli-
 * zing this code.
 *
 * Procedures and their roles:
 *****************************
 *
 * SetMesh: defines the process topology to be a 2-dimensional
 *          torus.
 * SetLand: defines a geometric partitioning of the problem.
 *          and initialises the local rabbit and fox populations.
 * SetComm: Defines the MPI data types for exchanging rows and
 *          columns among nearest neighbour processes.
 * Evolve:  Is called repeatedly to compute the next generations
 *          of both foxes and rabbits. Evolve calls the function
 *          FillBorder to exchange halo data between processes.
 * GetPopulation: Computes the total population of the specie
 *          it is given as argument.
 * FillBorder: Does the actual halo data swaps for the specie it
 *          is given has argument.
 *
 ***********************************************************************/

/* Constants and array bounds */
#include "param.h"

/* A few useful macros. */
#include "macro.h"

/* -- Arrays -- */
/* The arrays storing the number of animals in each stretch of land
 * have borders that are used as buffers for data from nearest neighbour
 * processors and to simplify the coding of periodic conditions. The
 * data assigned to the current process is stored in columns 1 through
 * LocalMaxY and in rows 1 through LocalMaxX of the two arrays below.
 * The halo data is stored in rows 0 and NS_Size+1 and in columns 0 and
 * WE_Size+1 of those arrays.
 */

float Rabbit[LocalMaxX+2][LocalMaxY+2];
float Fox[LocalMaxX+2][LocalMaxY+2];

/* The next two arrays are used in function Evolve() to compute
 * the next generations of rabbits and foxes.
 */
float TRabbit[LocalMaxX+2][LocalMaxY+2];
float TFox[LocalMaxX+2][LocalMaxY+2];

void main(argc, argv)
int  argc;
char *argv[];
{

    /* -- Local Arrays -- */
    /* The array named model is used to pass the model parameters
     * across procedures.
     */
    float model[2][3];

    /* The local indices of a stretch of land are the indices used by the
     * process assigned to that stretch of land. The global indices of the
     * same stretch of land are its local indices when only one process is
     * involved in the computation, i.e. when the rabbit and fox populations
     * are kept in one big array each. The array lbnds provides a mapping
     * between local and global indices, pictorially:
     *
     *                 lbnds[WE][0]   lbnds[WE][1]
     *                      |              |
     *                      V              V
     *                      ----------------
     *       lbnds[NS][0]-> |              |
     *                      |  Local data  |
     *                      |  of current  |
     *                      |   process    |
     *       lbnds[NS][1]-> |              |
     *                      ----------------
     *
     * The north-eastern most stretch of land assigned to the current process
     * has global indices (lbnds[NS][0], lbnds[WE][0]) and
     * the south-western most stretch of land assigned to the current process
     * has global indices (lbnds[NS][1],lbnds[WE][1]).
     */
    int   lbnds[NADIM][2];

    /* The ranks of the current process and of its nearest neighbours in
     * the 2-dimensional mesh are kept in the next array. Constants
     * HERE, NORTH, SOUTH, EAST and WEST are defined in the file param.h
     * to help use this array.
     */
    int neigh[1+2*NPDIM];

    /* Loop indices, bounds and population counters */
    int k, i;
    float nbrab, nbfox;

    /* MPI parameters and variables */
    int err;
    MPI_Comm comm;
    MPI_Datatype types[NADIM];

    /* Call MPI. */
    MPI_Init(&argc,&argv);

    /* Initialise the problem. */
    err = SetMesh(&comm);
    err = SetLand(Rabbit,Fox,lbnds,model,comm);
    err = SetComm(types,neigh,lbnds,comm);
 
    /* Iterate. */
    for( k=1; k<=NITER; k++) {
        err = Evolve(Rabbit,Fox,model,types,neigh,lbnds,comm); 
        if( !(k%PERIOD) ) {
            err = GetPopulation(Rabbit,lbnds,&nbrab,0,comm); 
            err = GetPopulation(Fox,lbnds,&nbfox,0,comm); 
            if( !(neigh[HERE]) )
                printf("Year %d: %.0f rabbits and %.0f foxes\n",
                   k, nbrab, nbfox);
        }
    }

    /* Close MPI. */
    MPI_Finalize();

}
/***********************************************************************
 * 
 * Set up a virtual topology for a 2-dimensional mesh of processes.
 * 
 ***********************************************************************/
int SetMesh(tcomm)
MPI_Comm *tcomm;
{
    int err;
    int npes;            /* the number of processes */
    int dims[NPDIM];     /* the number of processes in each dimension
                          * of the 2D mesh.
                          */
    int i;
    int periods[NPDIM];  /* Constants TRUE and FALSE are defined in param.h */
    int reorder;

    err = MPI_SUCCESS;

    /* Find out how many processes there are out there. */
    npes = 1;

    /* Set the dimensions of the process mesh. In the first instance,
     * mimic, say a npes X 1 ring of processes by setting dims[NS]=npes
     * and dims[WE]=1.
     */
    dims[NS] = 1;
    dims[WE] = 1;


    /* Make all dimensions of the virtual topology periodic. */



    /* Call MPI_Cart_create to set up a virtual topology. */



    /* Return the error code. */
    return(err);

}
/***********************************************************************
 * 
 * Compute the same initial populations of foxes and rabbits as the
 * sequential program does.
 * 
 ***********************************************************************/
int SetLand(Rabbit,Fox,lbnds,model,comm)
float    Rabbit[LocalMaxX+2][LocalMaxY+2];
float    Fox[LocalMaxX+2][LocalMaxY+2];
int      lbnds[NADIM][2]; /* see the defining comment in the main program. */
float    model[2][3];
MPI_Comm comm;
{
    int err;
    int gi, gj;
    int li, lj;
    int dx, dy; /* the largest number of rows and columns assigned to
                 * any process.
                 */
    int fx, lx; /* the global indices of the first and last row
                 * assigned to the current process.
                 */
    int fy, ly; /* similar to above. */
    int dims[NPDIM];   /* the number of processes in each dimension
                        * of the 2D mesh.
                        */
    int coords[NPDIM]; /* the current process has cartesian coordinates
                        * (coords[NS],coords[WE]).
                        */
    int periods[NPDIM];

    err = MPI_SUCCESS;

    /* Get process info. */


    /* Set the parameters of the predator-prey model. */
    model[RABBIT][SAME] = -0.2;
    model[RABBIT][OTHER] = 0.6;
    model[RABBIT][MIGRANT] = 0.01;
    model[FOX][OTHER] = 0.6;
    model[FOX][SAME] = -1.8;
    model[FOX][MIGRANT] = 0.02;

    /* Use a geometric decomposition and compute the global indices of the
     * first and last row of data on the current process. 
     */
    dx = NS_Size;
    fx = 1;
    lx = NS_Size;

    /* Check validity of global indices. */
    if( fx > NS_Size ) lx = fx - 1;
    else if( lx > NS_Size ) lx = NS_Size;

    /* Do the same with columns of data instead of rows. */
    dy = WE_Size;
    fy = 1;
    ly = WE_Size;

    /* Check validity of global indices. */
    if( fy > WE_Size ) ly = fy - 1;
    else if( ly > WE_Size ) ly = WE_Size;

    /* Check the local array Rabbit and Fox are large enough
     * to store the assigned rows and columns.
     */
    if(lx-fx+1>LocalMaxX || ly-fy+1>LocalMaxY) {
       fprintf(stderr,"Ran out of memory.\n");
       exit(1);
    }

    /* Fill the local arrays for foxes and rabbits. */
    for( gj=fy; gj<=ly; gj++) {
        lj = gj - fy + 1;
        for( gi=fx;  gi<=lx; gi++) {
            li = gi - fx + 1;
            Rabbit[li][lj] =
                128.0*(gi-1)*(NS_Size-gi)*(gj-1)*(WE_Size-gj) /
                (float)(NS_Size*NS_Size*WE_Size*WE_Size);
            Fox[li][lj] =
                8.0*(gi/(float)(NS_Size)-0.5)*(gi/(float)(NS_Size)-0.5)+ 
                8.0*(gj/(float)(WE_Size)-0.5)*(gj/(float)(WE_Size)-0.5);
        }
    }

    /* Save the lower and upper bounds of global array indices
     * for later use.
     */
    lbnds[NS][0] = fx;
    lbnds[NS][1] = lx;
    lbnds[WE][0] = fy;
    lbnds[WE][1] = ly;

/* Return the error code. */
    return(err);

}
/***********************************************************************
 * 
 * Initialise MPI data types and compute the ranks of nearest neighbours.
 * 
 ***********************************************************************/
int SetComm(types,neigh,lbnds,comm)
MPI_Datatype types[NADIM];
int          neigh[1+2*NPDIM];
int          lbnds[NADIM][2];
MPI_Comm     comm;
{
    int err;
    int coords[NPDIM];
    int lc;
    int lr;
    int i;

    err = MPI_SUCCESS;

    /* Find the rank of the current process. */
    neigh[HERE] = 0;

    /* Find the rank of its nearest neighbours. */
    neigh[NORTH] = neigh[SOUTH] = 0;
    neigh[WEST] = neigh[EAST] = 0;

    /* Compute the numbers of rows and columns assigned to the
     * current process.
     */
    lr = lbnds[NS][1] - lbnds[NS][0] + 1;
    lc = lbnds[WE][1] - lbnds[WE][0] + 1;

    /* Define a MPI datatype for moving one local column among
     * nearest neighbours.
     */
    types[COLUMN] = 0;



    /* Define a MPI datatype for moving one local row among 
     * nearest neighbours.
     */
    types[ROW] = 0;



    /* Return error code. */
    return(err);
}
/***********************************************************************
 * 
 * Compute the next generation of foxes and rabbits.
 * 
 ***********************************************************************/
int Evolve(Rabbit,Fox,model,types,neigh,lbnds,comm)
float        Rabbit[LocalMaxX+2][LocalMaxY+2];
float        Fox[LocalMaxX+2][LocalMaxY+2];
float        model[2][3];
MPI_Datatype types[NADIM];
int          neigh[1+2*NPDIM];
int          lbnds[NADIM][2];
MPI_Comm     comm;
{
    int err;
    int li, lj;
    int lr, lc;
    float AlR, BtR, MuR, AlF, BtF, MuF;

    err = MPI_SUCCESS;

    AlR = model[RABBIT][SAME];
    BtR = model[RABBIT][OTHER];
    MuR  = model[RABBIT][MIGRANT];
    BtF = model[FOX][SAME];
    AlF = model[FOX][OTHER];
    MuF  = model[FOX][MIGRANT];

    /* Fill-in the border of the local Rabbit array. */
    err = FillBorder(Rabbit,types,neigh,lbnds,comm); 

    /* Fill-in the border of the local Fox array. */
    err = FillBorder(Fox,types,neigh,lbnds,comm); 

    /* Compute the upper array bounds for the rows and columns
     * of the local Rabbit and Fox arrays.
     */
    lr = lbnds[NS][1] - lbnds[NS][0] + 1;
    lc = lbnds[WE][1] - lbnds[WE][0] + 1;

    /* Update the local population data. */
    for( lj=1; lj<=lc; lj++){
        for( li=1; li<=lr; li++){
            TRabbit[li][lj] = (1.0+AlR-4.0*MuR)*Rabbit[li][lj] +
                BtR*Fox[li][lj] +
                MuR*(Rabbit[li][lj-1]+Rabbit[li][lj+1]+
                Rabbit[li-1][lj]+Rabbit[li+1][lj]);
            TFox[li][lj] = AlF*Rabbit[li][lj] +
                (1.0+BtF-4.0*MuF)*Fox[li][lj] +
                MuF*(Fox[li][lj-1]+Fox[li][lj+1]+
                Fox[li-1][lj]+Fox[li+1][lj]);
        }
    }

    /* Ensure the numbers in Rabbit and Fox are non-negative. */
    for( lj=1; lj<=lc; lj++){
        for( li=1; li<=lr; li++){
            Rabbit[li][lj] = MAX( 0.0, TRabbit[li][lj]);
            Fox[li][lj] = MAX( 0.0, TFox[li][lj]);
        }
    }

    /* Return the error code. */
    return(err);
}
/***********************************************************************
 * 
 * Set the margin of one of the local data array. 
 * 
 ***********************************************************************/
int FillBorder(Animal,types,neigh,lbnds,comm)
float        Animal[LocalMaxX+2][LocalMaxY+2];
MPI_Datatype types[NADIM];
int          neigh[1+2*NPDIM];
int          lbnds[NADIM][2];
MPI_Comm     comm;
{
    int        err;

    int        lc;
    int        lr;
    int        i, j;

    err = MPI_SUCCESS;

    /* Compute the upper array bounds for the rows and columns
     * of the local Rabbit and Fox arrays.
     */
    lr = lbnds[NS][1] - lbnds[NS][0] + 1;
    lc = lbnds[WE][1] - lbnds[WE][0] + 1;

    /* Eastward and Westward data shifts. */
    if( lc == WE_Size ) {
       /* Current node has no distinct eastern or western neighbour. */
       for( i=1; i<=lr; i++) {
           Animal[i][0] = Animal[i][lc];
           Animal[i][lc+1] = Animal[i][1];
       }
    } else {
       /* Shift local borders eastward using a MPI_Send followed
        * by a MPI_Recv.
        */





       /* Shift local borders westward. */





    } 

    /* Northward and southward data shifts. */
    if( lr == NS_Size ) {
       /* The current node has no proper nearest neighbours to
        * the north and to the south.
        */
       for( j=1; j<=lc; j++) {
          Animal[0][j] = Animal[lr][j];
          Animal[lr+1][j] = Animal[1][j];
       }
     } else {
       /* Shift local borders northward. */





       /* Shift local borders southward. */




     }

    /* Return the error code. */
    return(err);
}
/***********************************************************************
 * 
 * Compute the number of individuals in a population.
 * 
 ***********************************************************************/
int GetPopulation(Animal,lbnds,tcount,root,comm)
float    Animal[LocalMaxX+2][LocalMaxY+2];
int      lbnds[NADIM][2];
float    *tcount;
int      root;
MPI_Comm comm;
{
    int   err;
    int   i, j, k, lr, lc;
    float p, q;

    err = MPI_SUCCESS;
    
    /* Compute the upper array bounds for the rows and columns
     * of the local Rabbit and Fox arrays.
     */
    lr = lbnds[NS][1] - lbnds[NS][0] + 1;
    lc = lbnds[WE][1] - lbnds[WE][0] + 1;

    /* Sum local population. */
    p = 0.0;
    for( j=1; j<=lc; j++)
      for( i=1; i<=lr; i++)
            p = p + Animal[i][j];

    /* Compute global population. */


    *tcount = p;

    /* Return the error code. */
    return(err);
}

