#include <stdio.h>
#include <mpi.h>
#include <math.h>

/***********************************************************************
 * 
 * This file contains C code for modelling a simple predator-prey model.
 * The two animal populations modelled are rabbits and foxes that live on
 * some piece of land. The animal populations are represented by two two-
 * dimensional arrays of size NS_Size by WE_Size. Precisely, the number of
 * foxes that live in the i,j-th stretch of land (with 1<=i<=NS_Size and
 * 1<=j<=WE_Size) is stored as Fox[i][j]. The corresponding figure for
 * rabbits is stored in Rabbit[i][j]. Boundary conditions are enforced so
 * that the land is actually a torus and can be thought to extend
 * periodically beyond the given bounds NS_Size and WE_Size.  The
 * populations of each stretch is updated at each generation according to
 * some simple rules and the total populations are summed at regular
 * intervals of time.
 * 
 * Processes are indexed by a 2-dimensional cartesian coordinate system 
 * and stretches of land are assigned to processes according to some 
 * geometric decomposition. The correspondence between array indices,
 * process coordinates and cardinal directions are shown in the diagrams
 * below with arrows pointing in the direction of increasing value.
 *
 *       +-----J----->       +-----Y----->
 *       |                   |              
 *       |                   |                  +--> East
 *       I  A[I][J]          X Proc[X][Y]       |        
 *       |                   |                  V 
 *       |                   |                South
 *       V                   V               
 *
 *
 *
 ***********************************************************************
 *
 * Procedures and their roles:
 *****************************
 *
 * SetMesh: defines the process topology to be a 2-dimensional
 *          torus.
 * SetLand: defines a geometric partitioning of the problem.
 *          and initialises the local rabbit and fox populations.
 * SetComm: Defines the MPI data types for exchanging rows and
 *          columns among nearest neighbour processes.
 * Evolve:  Is called repeatedly to compute the next generations
 *          of both foxes and rabbits. Evolve calls the function
 *          FillBorder to exchange halo data between processes.
 * GetPopulation: Computes the total population of the species
 *          it is given as argument.
 * FillBorder: Does the actual halo data swaps for the species it
 *          is given has argument.
 *
 ***********************************************************************/


/* Constants and array bounds */
#include "param.h"

/* A few useful macros. */
#include "macro.h"

/* Function Prototypes */
int SetMesh ( MPI_Comm * );
int SetLand ( float[LocalMaxX+2][LocalMaxY+2],
              float[LocalMaxX+2][LocalMaxY+2],
              int[NADIM][2],
              float[2][3],
              MPI_Comm);
int SetComm ( MPI_Datatype[NADIM],
              int[1+2*NPDIM],
              int[NADIM][2],
              MPI_Comm);
int Evolve ( float[LocalMaxX+2][LocalMaxY+2],
             float[LocalMaxX+2][LocalMaxY+2],
             float[2][3],
             MPI_Datatype[NADIM],
             int[1+2*NPDIM],
             int[NADIM][2],
             MPI_Comm);
int FillBorder ( float[LocalMaxX+2][LocalMaxY+2],
                 MPI_Datatype[NADIM],
                 int[1+2*NPDIM],
                 int[NADIM][2],
                 MPI_Comm);
int GetPopulation ( float[LocalMaxX+2][LocalMaxY+2],
                    int[NADIM][2],
                    float *,
                    int,
                    MPI_Comm);
int CHK( int err, char str[] );


/* Global Arrays */
/* The arrays storing the number of animals in each stretch of land
 * have borders that are used as buffers for data from nearest neighbour
 * processors and to simplify the coding of periodic conditions. The
 * data assigned to the current process is stored in columns 1 through
 * LocalMaxY and in rows 1 through LocalMaxX of the two arrays below.
 * The halo data is stored in rows 0 and NS_Size+1 and in columns 0 and
 * WE_Size+1 of those arrays.
 */
float Rabbit[LocalMaxX+2][LocalMaxY+2];
float Fox[LocalMaxX+2][LocalMaxY+2];

/* The next two arrays are used in function Evolve() to compute
 * the next generations of rabbits and foxes.
 */
float TRabbit[LocalMaxX+2][LocalMaxY+2];
float TFox[LocalMaxX+2][LocalMaxY+2];

void main(int argc, char *argv[])
{

    /* -- Local Arrays -- */
    /* The array named model is used to pass the model parameters
     * across procedures.
     */
    float model[2][3];

    /* The local indices of a stretch of land are the indices used by the
     * process assigned to that stretch of land. The global indices of the
     * same stretch of land are its local indices when only one process is
     * involved in the computation, i.e. when the rabbit and fox populations
     * are kept in one big array each. The array lbnds provides a mapping
     * between local and global indices, pictorially:
     * 
     *                 lbnds[WE][0]   lbnds[WE][1]
     *                      |              |
     *                      V              V
     *                      ----------------
     *       lbnds[NS][0]-> |              |
     *                      |  Local data  |
     *                      |  of current  |
     *                      |   process    |
     *       lbnds[NS][1]-> |              |
     *                      ----------------
     *
     * The north-eastern most stretch of land assigned to the current process
     * has global indices (lbnds[NS][0], lbnds[WE][0]) and
     * the south-western most stretch of land assigned to the current process
     * has global indices (lbnds[NS][1],lbnds[WE][1]).
     */

    int   lbnds[NADIM][2];

    /* The ranks of the current process and of its nearest neighbours in
     * the 2-dimensional mesh are kept in the next array. Constants
     * HERE, NORTH, SOUTH, EAST and WEST are defined in the file param.h
     * to help use this array.
     */
    int neigh[1+2*NPDIM];

    /* Loop indices, bounds and population counters */
    int k;
    float nbrab, nbfox;

    /* MPI parameters and variables */
    int err;
    MPI_Comm comm;
    MPI_Datatype types[NADIM];

    /* Call MPI. */
    MPI_Init(&argc,&argv); 

    /* Initialise the problem. */
    SetMesh(&comm);
    SetLand(Rabbit,Fox,lbnds,model,comm); 
    SetComm(types,neigh,lbnds,comm);
 
    /* Iterate. */
    for( k=1; k<=NITER; k++) {
        Evolve(Rabbit,Fox,model,types,neigh,lbnds,comm); 
        if( !(k%PERIOD) ) {
            GetPopulation(Rabbit,lbnds,&nbrab,0,comm); 
            GetPopulation(Fox,lbnds,&nbfox,0,comm); 
            if( !(neigh[HERE]) )
                printf("Year %d: %.0f rabbits and %.0f foxes\n",
                   k, nbrab, nbfox);
        }
    }

    /* Close MPI. */
    MPI_Finalize(); 

}
/***********************************************************************
 * 
 * Set up a virtual topology for a 2-dimensional mesh of processes.
 * 
 ***********************************************************************/
int SetMesh( MPI_Comm *tcomm )
{
    int err;
    int npes;            /* the number of processes */
    int dims[NPDIM];     /* the number of processes in each dimension
                          * of the 2D mesh.
                          */
    int i;
    int periods[NPDIM];  /* Constants TRUE and FALSE are defined in param.h */
    int reorder;

    err = MPI_SUCCESS;

    /* Find out how many processes there are out there. */
    MPI_Comm_size(MPI_COMM_WORLD,&npes); 

    /* Set the dimensions of the process mesh. In the first instance,
     * mimic a 1Xnpes ring of processes by setting dims[NS]=1 and
     * dims[WE]=npes.
     */
    for(i=0;i<NPDIM;dims[i++]=0);
    MPI_Dims_create(npes,NPDIM,dims); 

    /* Make all dimensions of the virtual topology periodic. */
    for( i=0; i<NPDIM; periods[i++]=TRUE);
    reorder = TRUE;

    /* Call MPI_Cart_create to set up a virtual topology. */
    MPI_Cart_create(MPI_COMM_WORLD, NPDIM, dims, periods,
        reorder, tcomm);

    /* Return the error code. */
    return(err);

}
/***********************************************************************
 * 
 * Initialise the populations of foxes and rabbits in the same way
 * as the sequential program does.
 * 
 ***********************************************************************/
int SetLand( float    Rabbit[LocalMaxX+2][LocalMaxY+2],
             float    Fox[LocalMaxX+2][LocalMaxY+2],
             int      lbnds[NADIM][2],
             float    model[2][3],
             MPI_Comm comm )
{
    int err;
    int gi, gj;
    int li, lj;
    int dx, dy;
    int fx, fy;
    int lx, ly;
    int dims[NPDIM];   /* see the comments in SetMesh() above. */
    int coords[NPDIM]; /* the current process has cartesian coordinates
                        * (coords[NS],coords[WE]).
                        */
    int periods[NPDIM];

    err = MPI_SUCCESS;

    /* Get process info. */
    MPI_Cart_get(comm, NPDIM, dims, periods, coords); 

    /* Set the parameters of the predator-prey model. */
    model[RABBIT][SAME] = -0.2;
    model[RABBIT][OTHER] = 0.6;
    model[RABBIT][MIGRANT] = 0.01;
    model[FOX][OTHER] = 0.6;
    model[FOX][SAME] = -1.8;
    model[FOX][MIGRANT] = 0.02;

    /* Compute the global indices of the first and last row of current
     *  process. 
     */
    dx = (NS_Size+dims[NS]-1) / dims[NS];
    fx = coords[NS]*dx + 1;
    lx = fx + dx - 1;
    if( fx > NS_Size ) lx = fx - 1;
    else if( lx > NS_Size ) lx = NS_Size;

    /* Do the same with columns instead of rows. */
    dy = (WE_Size+dims[WE]-1) / dims[WE];
    fy = coords[WE]*dy + 1;
    ly = fy + dy - 1;
    if( fy > WE_Size ) ly = fy - 1;
    else if( ly > WE_Size ) ly = WE_Size;

    /* Check the local array Rabbit and Fox are large enough
     * to store the assigned rows and columns.
     */
    if((lx-fx+1)>LocalMaxX || (ly-fy+1)>LocalMaxY) {
       fprintf(stderr,"Ran out of memory\n");
       exit(1);
    }

    /* Fill the local arrays for foxes and rabbits. */
    for( gj=fy; gj<=ly; gj++) {
        lj = gj - fy + 1;
        for( gi=fx;  gi<=lx; gi++) {
            li = gi - fx + 1;
            Rabbit[li][lj] =
                128.0*(gi-1)*(NS_Size-gi)*(gj-1)*(WE_Size-gj) /
                (float)(NS_Size*NS_Size*WE_Size*WE_Size);
            Fox[li][lj] =
                8.0*(gi/(float)(NS_Size)-0.5)*(gi/(float)(NS_Size)-0.5)+ 
                8.0*(gj/(float)(WE_Size)-0.5)*(gj/(float)(WE_Size)-0.5);
        }
    }

    /* Save the lower and upper bounds of global array indices
     * for later use.
     */
    lbnds[NS][0] = fx;
    lbnds[NS][1] = lx;
    lbnds[WE][0] = fy;
    lbnds[WE][1] = ly;

/* Return the error code. */
    return(err);

}
/***********************************************************************
 * 
 * Initialise MPI data types and compute the ranks of nearest neighbours.
 * 
 ***********************************************************************/
int SetComm( MPI_Datatype types[NADIM],
             int          neigh[1+2*NPDIM],
             int          lbnds[NADIM][2],
             MPI_Comm     comm )
{
    int err;
    int lc;
    int lr;

    err = MPI_SUCCESS;

    /* Find the rank of the current process. */
    MPI_Comm_rank(comm,&neigh[HERE]); 

    /* Find the rank of its nearest neighbours. If you have set
     * dims[NS]=1 in SetComm above it may be safer to 
     * specify neigh[NORTH]=neigh[SOUTH]=neigh[HERE] at this point.
     */
    MPI_Cart_shift(comm,NS,1,&neigh[NORTH],&neigh[SOUTH]);
    MPI_Cart_shift(comm,WE,1,&neigh[EAST],&neigh[WEST]);

    /* Define a MPI datatype for moving one local column among
     * nearest neighbours.
     */
    lr = lbnds[NS][1] - lbnds[NS][0] + 1;
    MPI_Type_vector(lr,1,LocalMaxY+2,MPI_FLOAT,&types[COLUMN]);
    MPI_Type_commit(&types[COLUMN]); 

    /* Define a MPI datatype for moving one local row among 
     * nearest neighbours.
     */
    lc = lbnds[WE][1] - lbnds[WE][0] + 1;
    MPI_Type_vector(1,lc,LocalMaxY+2,MPI_FLOAT,&types[ROW]);
    MPI_Type_commit(&types[ROW]); 

    /* Return error code. */
    return(err);
}
/***********************************************************************
 * 
 * Compute the next generation of foxes and rabbits.
 * 
 ***********************************************************************/
int Evolve( float        Rabbit[LocalMaxX+2][LocalMaxY+2],
            float        Fox[LocalMaxX+2][LocalMaxY+2],
            float        model[2][3],
            MPI_Datatype types[NADIM],
            int          neigh[1+2*NPDIM],
            int          lbnds[NADIM][2],
            MPI_Comm     comm )
{
    int err, li, lj, lr, lc;
    float AlR, BtR, MuR, AlF, BtF, MuF;

    err = MPI_SUCCESS;

    AlR = model[RABBIT][SAME];
    BtR = model[RABBIT][OTHER];
    MuR  = model[RABBIT][MIGRANT];
    BtF = model[FOX][SAME];
    AlF = model[FOX][OTHER];
    MuF  = model[FOX][MIGRANT];

    /* Fill-in the border of the local Rabbit array. */
    FillBorder(Rabbit,types,neigh,lbnds,comm); 

    /* Fill-in the border of the local Fox array. */
    FillBorder(Fox,types,neigh,lbnds,comm); 

    /* Compute the upper array bounds for the rows and columns
     * of the local Rabbit and Fox arrays.
     */
    lr = lbnds[NS][1] - lbnds[NS][0] + 1;
    lc = lbnds[WE][1] - lbnds[WE][0] + 1;

    for(lj=0; lj<=lc+1; lj++)
       for(li=0; li<=lr+1;li++)
          if(ABS(Rabbit[li][lj])>500 ||
          ABS(Fox[li][lj])>500) {
            fprintf(stderr,"Garbbage in Rabbit of Fox.\n");
            MPI_Abort(MPI_COMM_WORLD, MPI_SUCCESS + 1);
          }
    /* Update the local population data. */
    for( lj=1; lj<=lc; lj++){
        for( li=1; li<=lr; li++){
            TRabbit[li][lj] = (1.0+AlR-4.0*MuR)*Rabbit[li][lj] +
                BtR*Fox[li][lj] +
                MuR*(Rabbit[li][lj-1]+Rabbit[li][lj+1]+
                Rabbit[li-1][lj]+Rabbit[li+1][lj]);
            TFox[li][lj] = AlF*Rabbit[li][lj] +
                (1.0+BtF-4.0*MuF)*Fox[li][lj] +
                MuF*(Fox[li][lj-1]+Fox[li][lj+1]+
                Fox[li-1][lj]+Fox[li+1][lj]);
        }
    }

    /* Ensure the numbers in Rabbit and Fox are non-negative. */
    for( lj=1; lj<=lc; lj++){
        for( li=1; li<=lr; li++){
            Rabbit[li][lj] = MAX( 0.0, TRabbit[li][lj]);
            Fox[li][lj] = MAX( 0.0, TFox[li][lj]);
        }
    }

    /* Return the error code. */
    return(err);
}
/***********************************************************************
 * 
 * Set the margin of one of the local data array. 
 * 
 ***********************************************************************/
int FillBorder( float        Animal[LocalMaxX+2][LocalMaxY+2],
                MPI_Datatype types[NADIM],
                int          neigh[1+2*NPDIM],
                int          lbnds[NADIM][2],
                MPI_Comm     comm )
{
    int        err;
    MPI_Status status;
    int        lc;
    int        lr;
    int        i, j;

    err = MPI_SUCCESS;

    /* Compute the upper array bounds for the rows and columns
     * of the local Rabbit and Fox arrays.
     */
    lc = lbnds[WE][1] - lbnds[WE][0] + 1;
    lr = lbnds[NS][1] - lbnds[NS][0] + 1;

    /* Eastward and Westward data shifts. */
    if( lc == WE_Size ) {
       /* Current node has no distinct eastern or western neighbour. */
       for( i=1; i<=lr; i++) {
           Animal[i][0] = Animal[i][lc];
           Animal[i][lc+1] = Animal[i][1];
       }
    } else if ( lc > 0 ) {
       /* Shift local borders eastward using a MPI_Send followed
        * by a MPI_Recv.
        */
       MPI_Sendrecv(&Animal[1][1],1,types[COLUMN],neigh[EAST],EAST,
             &Animal[1][lc+1],1,types[COLUMN],neigh[WEST],EAST,comm,&status);

       /* Shift local borders westward. */
       MPI_Sendrecv(&Animal[1][lc],1,types[COLUMN],neigh[WEST],WEST,
             &Animal[1][0],1,types[COLUMN],neigh[EAST],WEST,comm,&status);

    } else {
       /* Current process is assigned no piece of land so it only
        * shifts around the data received by its nearest neighbours.
        */

       /* Shift data coming from the east. */
       MPI_Recv(&Animal[1][lc+1],1,types[COLUMN],neigh[WEST],
           EAST,comm,&status);
       MPI_Send(&Animal[1][1],1,types[COLUMN],neigh[EAST],
           EAST,comm);

       /* Shift data coming from the west. */
       MPI_Recv(&Animal[1][0],1,types[COLUMN],neigh[EAST],
           WEST,comm,&status);
       MPI_Send(&Animal[1][lc],1,types[COLUMN],neigh[WEST],
           WEST,comm);
    }

    /* Northward and southward data shifts. */
    if( lr == NS_Size ) {
       /* The current node has no proper nearest neighbours to
        * the north and to the south.
        */
       for( j=1; j<=lc; j++) {
          Animal[0][j] = Animal[lr][j];
          Animal[lr+1][j] = Animal[1][j];
       }
     } else if( lr > 0 ) {
       /* Shift local borders northward. */
       MPI_Sendrecv(&Animal[1][1],1,types[ROW],neigh[NORTH],NORTH,
             &Animal[lr+1][1],1,types[ROW],neigh[SOUTH],NORTH,comm,&status);

       /* Shift local borders southward. */
       MPI_Sendrecv(&Animal[lr][1],1,types[ROW],neigh[SOUTH],SOUTH,
             &Animal[0][1],1,types[ROW],neigh[NORTH],SOUTH,comm,&status);
     } else {
       /* The current node has no real work assigned to it. */
       /* Shift data received from the south. */
       MPI_Recv(&Animal[lr+1][1],1,types[ROW],neigh[SOUTH],
           NORTH,comm,&status);
       MPI_Send(&Animal[1][1],1,types[ROW],neigh[NORTH],
           NORTH,comm);

       /* Shift data received from the north. */
       MPI_Recv(&Animal[0][1],1,types[ROW],neigh[NORTH],
           SOUTH,comm,&status);
       MPI_Send(&Animal[lr][1],1,types[ROW],neigh[SOUTH],
           SOUTH,comm); 
     }

    /* Return the error code. */
    return(err);
}
/***********************************************************************
 * 
 * Compute the number of individuals in a population.
 * 
 ***********************************************************************/
int GetPopulation( float    Animal[LocalMaxX+2][LocalMaxY+2],
                   int      lbnds[NADIM][2],
                   float    *tcount,
                   int      root,
                   MPI_Comm comm )
{
    int   err;
    int   i, j, lr, lc;
    float p, q;

    err = MPI_SUCCESS;
    
    /* Compute the upper array bounds for the rows and columns
     * of the local Rabbit and Fox arrays.
     */
    lr = lbnds[NS][1] - lbnds[NS][0] + 1;
    lc = lbnds[WE][1] - lbnds[WE][0] + 1;

    p = 0.0;
    for( j=1; j<=lc; j++)
      for( i=1; i<=lr; i++)
            p = p + Animal[i][j];

    MPI_Reduce(&p,&q,1,MPI_FLOAT,MPI_SUM,root,comm); 

    *tcount = q;

    /* Return the error code. */
    return(err);
}
/***********************************************************************
 * 
 * Check for a non-zero MPI error code and print error message if
 * necessary.
 * 
 ***********************************************************************/
int CHK( int err, char str[] )
{
    int  rlen; 
    int  jerror;
    char msg[MPI_MAX_ERROR_STRING];

    if( err != MPI_SUCCESS ) {
        jerror = MPI_Error_string(err,msg,&rlen);
        fprintf(stderr,"CHK(%s):%s\n",str, msg);
        jerror = MPI_Abort(MPI_COMM_WORLD,err);
    }

    return(jerror);
}
